package easik.sketch.util.Export.Constraints;

import java.util.ArrayList;
import java.util.HashMap;

import easik.sketch.util.Export.Components.ExportPath;

/**
 * General class to hold a constraint generated directly from the XML file.  Constraints are never 
 * formed directly from this class, thus the type of constraint is determined by the creating subclass.
 * Should not be used for any other purpose than database exportation, as it 
 * contains no graphical information.
 * 
 * @author Vera Ranieri 2006
 * @since 2006-05-22 Vera Ranieri
 * @version 2006-07-07 Vera Ranieri
 *
 */
public class ExportConstraint {
	/**
	 * The paths involved in this constraint
	 */
	protected ArrayList<ExportPath> _paths;
	/**
	 * The procedures created by the constraint
	 */
	protected ArrayList<String> _procedureStrings;
	/**
	 * The base table of the constraint
	 */
	protected String _baseTable;

	/**
	 * A unique identifier of the constraint
	 */
	protected int _id;
	
	/**
	 * HashMap of all procedures that are called <b>BEFORE INSERT</b>, indexed by the table that it applies to
	 */
	protected HashMap<String, String> _b_i_tables;
	/**
	 * HashMap of all procedures that are called <b>AFTER INSERT</b>, indexed by the table that it applies to
	 */
	protected HashMap<String, String> _a_i_tables;
	/**
	 * HashMap of all procedures that are called<b>BEFORE DELETE</b>, indexed by the table that it applies to
	 */
	protected HashMap<String, String> _b_d_tables;
	/**
	 * The database name
	 */
	protected String _db_name;
	
	/**
	 * Constructs a new contraint for exportation.  Overloaded constructor that is 
	 * never called explicitly.
	 */
	public ExportConstraint(){
		
		_paths = new ArrayList<ExportPath>();
		_procedureStrings = new ArrayList<String>();
		_b_i_tables = new HashMap<String, String>();
		_a_i_tables = new HashMap<String, String>();
		_b_d_tables = new HashMap<String, String>();
	}
	
	/**
	 * Add a path to this constraint.
	 * 
	 * @param p The path that participates in the constraint
	 */
	public void addPath(ExportPath p){
		_paths.add(p);
	}
	
	/**
	 * Get all paths participating in this constraint
	 * @return The paths associated with this constraint
	 */
	public ArrayList<ExportPath> getPaths(){
		return _paths;
	}
	
	/**
	 * Sets the base table for the constraint.
	 * Empty method that is overloaded by extending classes.
	 */
	public void setBaseTable(){};
	
	/**
	 * Sets the target table(s) for the constraint.
	 * Empty method that is overloaded by extending classes.
	 * 
	 * @since 2006-06-22 Vera Ranieri
	 */
	public void setTargetTable(){};
	/**
	 * The table that serves as a base for the constraint.  Set differently depending on the type 
	 * of constraint.
	 * 
	 * @return The name of the base table
	 */
	public String getBaseTable(){
		return _baseTable;
	}
	
	/**
	 * Sets the constraint string for this constraint.  Overloaded by all subclasses
	 * 
	 * @since 2006-06-22 Vera Ranieri
	 */
	public void setConstraintStrings(){};
	
	/**
	 * Gets the constraint string for this constraint.  Overloaded by all subclasses.
	 * 
	 * @return The constraint strings formatted for sql, in an array list.
	 * @since 2006-06-09 Vera Ranieri
	 */
	public ArrayList<String> getConstraintStrings(){
		return _procedureStrings;
	}
	
	/**
	 * Gets the hash map containing all procedures to be called at <it>time</it>. 
	 * 
	 * @param time The time for which the constraints should be called.
	 * @return HashMap of all tables to be called at <it>time</it>, referenced by the table to which they apply
	 * @since 2006-06-28 Vera Ranieri
	 */
	public HashMap<String, String> getTimeNames(String time){
		if(time.equals(ExportTriggerFormatter.BEFORE_INSERT_TIME))
			return _b_i_tables;
		else if(time.equals(ExportTriggerFormatter.AFTER_INSERT_TIME))
			return _a_i_tables;
		else 
			return _b_d_tables;
		
	}

	/**
	 * Sets the database name
	 * 
	 * @param db_name The database name to set.
	 */
	public void set_db_name(String db_name) {
		_db_name = db_name;
	}
	
}
